(*************************************************************************
 *  BackgroundEngineU.pas                                                *
 *  Vladimr Slvik 2008-10                                              *
 *  Delphi 7 Personal                                                    *
 *  cp1250                                                               *
 *                                                                       *
 *  Background generator - automatically generates and caches            *
 *  backgrounds for main engine. Lazy loading and invalidations.         *
 *                                                                       *
 *  -additional libraries: Graphics32                                    *
 *************************************************************************)

unit BackgroundEngineU;

{$INCLUDE ..\Switches.inc}
{t default -}

//------------------------------------------------------------------------------
interface

uses Types, Classes,
     GR32,
     ClassBaseU, CoreTypeU, ConfigU;

//------------------------------------------------------------------------------

const bsUndefined = -1; bsColor = 0; bsPicture = 1;

type TBackgroundSetting = bsUndefined .. bsPicture;

//------------------------------------------------------------------------------
{
  This class acts as a convenient "insulation" of from background logic; it
  provides inputs - color, picture and settings - and outputs - buffers ready
  for use in the main engine.

  Another concept used here is lazy updating; buffers are (re)generated after
  first request.
}

type TBackgroundEngine = class
       constructor Create;
       destructor Destroy; override;
       procedure AfterConstruction; override;
     private
       FPicture: TBitmap32;
       // source picture
       FColor: TColor32;
       // source color
       FPctBackNorm: TBitmap32;
       FPctBackDark: TBitmap32;
       FPctBackHigh: TBitmap32;
       // background caches
       FOnChange: TNotifyEvent;
       FSize: TRect;
       FSetting: TBackgroundSetting;
       FLookupTable: array [TViewMode] of TBitmap32;
       FReadyTable: array [TViewMode] of Boolean;
       FUpdateProcTable: array [TViewMode] of procedure of object;
       // The whole "background" concept can be organized into table since it's
       // "indexed" by view mode and color|picture choice. Organizing in tables
       // should also speed up lookups...
       // It is important to note that these tables contain only pointers, not
       // real data!
       FOptions: TSdiEdOptions;
       procedure SetColor(const AColor: TColor32);
       procedure SetSetting(const ASetting: TBackgroundSetting);
       procedure SetOptions(const AOptions: TSdiEdOptions);
       // setters
       procedure ReloadLookupTable;
       procedure InvalidateReadyTable;
       procedure InvalidateReadyTableNight;
       procedure InvalidateReadyTableHighlight;
       procedure PrepareUpdateTable;
       // table stuff
       procedure UpdateDayPic;
       procedure UpdateNightPic;
       procedure UpdateHighlightPic;
       // updaters
       procedure CallBack;
       // tell the main engine about change
       procedure PictureChanged(Sender: TObject);
       procedure SetSize(ASize: TRect);
     public
       property Options: TSdiEdOptions read FOptions write SetOptions;
       procedure Render(const Target: TBitmap32; const ViewMode: TViewMode;
         const DrawMode: TDrawMode; const Area: TRect);
     published
       property Picture: TBitmap32 read FPicture;
       property Color: TColor32 read FColor write SetColor;
       property Setting: TBackgroundSetting read FSetting write SetSetting;
       property Size: TRect read FSize write SetSize;
       property OnChange: TNotifyEvent read FOnChange write FOnChange;
     end;

//==============================================================================
implementation

uses SysUtils, CalcUtilU, CoreLowU;
//------------------------------------------------------------------------------

procedure TBackgroundEngine.PictureChanged(Sender: TObject);
begin
  if Sender = FPicture then begin
    InvalidateReadyTable;
    CallBack;
  end {$IFDEF DEBUG} else raise Exception.Create('Wrong picture!') {$ENDIF};
end;

//------------------------------------------------------------------------------

procedure TBackgroundEngine.CallBack;
begin
  if Assigned(FOnChange) then FOnChange(Self);
end;

//------------------------------------------------------------------------------

procedure TBackgroundEngine.UpdateDayPic;
// day picture = tile "as is"
begin
  FPctBackNorm.SetSize(RectWidth(FSize), RectHeight(FSize));
  TileBitmap32To(FPicture, FPctBackNorm);
  FReadyTable[vmNormal]:= True;
end;

//------------------------------------------------------------------------------

procedure TBackgroundEngine.UpdateNightPic;
// night picture = copy & convert
var B: TBitmap32;
begin
  B:= TBitmap32.Create;
  B.Assign(FPicture);
  MakeDarkened(B, FOptions, B.BoundsRect);
  FPctBackDark.SetSize(RectWidth(FSize), RectHeight(FSize));
  TileBitmap32To(B, FPctBackDark);
  B.Free;
  FReadyTable[vmDark]:= True;
end;

//------------------------------------------------------------------------------

procedure TBackgroundEngine.UpdateHighlightPic;
// highlighted picture = copy & convert
var B: TBitmap32;
begin
  B:= TBitmap32.Create;
  B.Assign(FPctBackNorm);
  MakeHighlighted(B, FOptions, B.BoundsRect);
  FPctBackHigh.SetSize(RectWidth(FSize), RectHeight(FSize));
  TileBitmap32To(B, FPctBackHigh);
  B.Free;
  FReadyTable[vmHighlight]:= True;
end;

//------------------------------------------------------------------------------

procedure TBackgroundEngine.SetOptions(const AOptions: TSdiEdOptions);
var Diff: TOptionsCompare;
begin
  Diff:= DiffOptions(FOptions, AOptions);
  if Diff.NightChanged then InvalidateReadyTableNight;
  if Diff.HighlightChanged then InvalidateReadyTableHighlight;
  FOptions:= AOptions;
  // Do assign only here, because before this we might eventually theoretically
  // in the future (yeah right :P ) want to get some more concise info about
  // changes and that needs two sets of data for comparison.
  CallBack; // tell daddy (main engine) that we have a new background to draw
end;

//------------------------------------------------------------------------------

procedure TBackgroundEngine.SetSetting(const ASetting: TBackgroundSetting);
begin
  FSetting:= ASetting;
  CallBack;
end;

//------------------------------------------------------------------------------

procedure TBackgroundEngine.SetColor(const AColor: TColor32);
begin
  FColor:= AColor;
  CallBack;
end;

//------------------------------------------------------------------------------

procedure TBackgroundEngine.InvalidateReadyTableNight;
begin
  FReadyTable[vmDark]:= False;
end;

//------------------------------------------------------------------------------

procedure TBackgroundEngine.InvalidateReadyTableHighlight;
begin
  FReadyTable[vmHighlight]:= False;
end;

//------------------------------------------------------------------------------

procedure TBackgroundEngine.InvalidateReadyTable;
var i: Integer;
begin
  for i:= vmUndefined to vmHighlight do
    FReadyTable[i]:= False;
end;

//------------------------------------------------------------------------------

procedure TBackgroundEngine.PrepareUpdateTable;
var i: Integer;
begin
  for i:= vmUndefined to vmHighlight do
    FUpdateProcTable[i]:= nil; // pre-fill with empty
  FUpdateProcTable[vmNormal]:= UpdateDayPic;
  FUpdateProcTable[vmDark]:= UpdateNightPic;
  FUpdateProcTable[vmHighlight]:= UpdateHighlightPic;
end;

//------------------------------------------------------------------------------

procedure TBackgroundEngine.ReloadLookupTable;
var i: Integer;
begin
  for i:= vmUndefined to vmHighlight do
    FLookupTable[i]:= nil; // pre-fill with empty
  FLookupTable[vmNormal]:= FPctBackNorm;
  FLookupTable[vmDark]:= FPctBackDark;
  FLookupTable[vmHighlight]:= FPctBackHigh;
end;

//------------------------------------------------------------------------------

constructor TBackgroundEngine.Create;
begin
  FPicture:= TBitmap32.Create;
  // main object
  FPctBackNorm:= TBitmap32.Create;
  FPctBackDark:= TBitmap32.Create;
  FPctBackHigh:= TBitmap32.Create;
  // cache
  ReloadLookupTable;
  PrepareUpdateTable;
  InvalidateReadyTable;
  FColor:= $FFE7FFFF;
end;

//------------------------------------------------------------------------------

procedure TBackgroundEngine.AfterConstruction;
begin
  FPicture.OnChange:= Self.PictureChanged;
end;

//------------------------------------------------------------------------------

destructor TBackgroundEngine.Destroy;
var B: TBitmap32;
    i: Integer;
begin
  for i:= vmUndefined to vmHighlight do begin
    B:= FLookupTable[i];
    if Assigned(B) then B.Free; // easier than accessing them all by name...
  end;
end;

//------------------------------------------------------------------------------

procedure TBackgroundEngine.SetSize(ASize: TRect);
begin
  if ((ASize.Left <> FSize.Left) or (ASize.Top <> FSize.Top) or
      (ASize.Bottom <> FSize.Bottom) or (ASize.Right <> FSize.Right)) then begin
    // argh these rects will kill me
    FSize:= ASize;
    InvalidateReadyTable;
  end;
end;

//------------------------------------------------------------------------------

procedure TBackgroundEngine.Render(const Target: TBitmap32; const ViewMode: TViewMode;
  const DrawMode: TDrawMode; const Area: TRect);
var B: TBitmap32; // just a reference
    C: TColor32;
begin
  if FSetting = bsColor then begin
    case ViewMode of
      vmNormal, vmUndefined: C:= FColor;
      vmDark: C:= DarkenedColorDumb(FColor, FOptions) or $FF000000;
      vmHighlight: C:= HighlightedColorDumb(FColor, FOptions) or $FF000000;
      else C:= FColor;
    end;
    with Area do Target.FillRect(Left, Top, Right, Bottom, C);
  end else begin
    B:= FLookupTable[ViewMode];
    if (B <> nil) then begin
      if not FReadyTable[ViewMode] then FUpdateProcTable[ViewMode];
      // if needed, update the requested bitmap
      B.DrawMode:= DrawMode;
      Target.Draw(Area, Area, B);
    end {$IFDEF DEBUG} else raise Exception.Create('Invalid index in TBackgroundEngine.Render.') {$ENDIF};
  end;
end;

//------------------------------------------------------------------------------

end.
