(*************************************************************************
 *  MainU.pas + MainU.dfm                                                *
 *  Vladimr Slvik 2005-10                                              *
 *  Delphi 7 Personal                                                    *
 *  cp1250                                                               *
 *                                                                       *
 *  main form of Shades :                                                *
 *    viewport, toolbars etc.                                            *
 *                                                                       *
 *  -additional libraries: Graphics32, StatusBarPro, PNGImage            *
 *  -you will need also SpinButton from sample Delphi components         *
 *************************************************************************)

unit MainU;

{$INCLUDE Switches.inc}
{t default -}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ImgList, ToolWin, StdCtrls, ComCtrls, ExtDlgs, ExtCtrls, Menus, // Delphi
  Spin, // Delphi sample components
  GR32, GR32_Image, GR32_Layers, GR32_ExtImage, // Graphics32
  SBPro, // StatusBarPro
  BrushButton, OptionU, CbFrameU, UndoU, CoreTypeU, ConfigU, ClassBaseU, CoreEngineU; // this project

{
 Before you mess with form and look, read...

 The whole window is filled with autosizing controls - understand how "Align"
 works.

 Toolbars are all aligned to top, switching the lower one is just hiding one and
 showing another. This may produce redraw flickers, "MainForm.DoubleBuffered"
 did not help much. I guess this comes from the fact that gOut tries to spread
 over the space freed by hidden toolbar and then again has to shrink when the
 new toolbar appears. I could not find anything like "BeginUpdate" or such for
 TForm. Maybe play with WM_paint and updating?

 Checkboxes on toolbar are a bit "hacked"; If you put a CB on TToolBar, it
 changes its height to match the toolbar. There was no reasonable way to put
 there two of them vertically, so I made a frame with them and put that construct
 on toolbar. Do not even touch it with mouse - as the form is packed with
 draggable controls (ok, 100% of its surface) and these two CBs are nested in
 3rd level, you will drag them once away and may not be able to get them back,
 causing further damage to form layout. Quite annoying, because then you have
 to load backup file ...

 The component for showing color under mouse position is also quite a bit nasty
 piece. It seems that TBitmap32 does not work properly with TStatusBarPro as
 owner - it does not move around with it, meaning eg. when you resize the form
 it stays where it is, and due to resizing it gets under viewport. Used
 workaround is, it is on the form itself, moved to front and Anchors are set so
 that it stays where it should. (Strange thing, because the TShape I used there
 before worked always well. This probably means the problem is not in the
 SBPro?)
}

type
  TMainFrm = class(TForm) //----------------------------------------------------
    tbControl: TToolBar;
    tbtnOpen: TToolButton;
    tbtnSave: TToolButton;
    tbtnSaveAs: TToolButton;
    ToolButton1: TToolButton;
    tbtnDayView: TToolButton;
    tbtnNightView: TToolButton;
    ToolButton3: TToolButton;
    tbtnSpecView: TToolButton;
    tbtnAHelp: TToolButton;
    ImageList: TImageList;
    gOut: TImgView32;
    pmHelp: TPopupMenu;
    pmiHelp: TMenuItem;
    pmiAbout: TMenuItem;
    ToolButton6: TToolButton;
    sbtnZoom: TSpinButton;
    tbtnOptions: TToolButton;
    tbPaint: TToolBar;
    StatusBar: TStatusBarPro;
    lbPosData: TLabel;
    lbZoom: TLabel;
    lbEditData: TLabel;
    tbtnUndo: TToolButton;
    i32ColorView: TImage32;
    tbtnPen: TToolButton;
    tbtnFloodFill: TToolButton;
    tbtnRedo: TToolButton;
    ToolButton12: TToolButton;
    memHint: TMemo;
    tbtnPxEraser: TToolButton;
    tbtnClose: TToolButton;
    tbtnFillEraser: TToolButton;
    tbtnBackground: TToolButton;
    pmOpenSpecial: TPopupMenu;
    pmiAboutBlank: TMenuItem;
    pmiAboutTest: TMenuItem;
    tbColors: TToolBar;
    i32ClrSel: TImage32;
    ToolButton10: TToolButton;
    tbtnFixBackground: TToolButton;
    CbFrame: TCbFrame;
    pnlTools: TPanel;
    Bevel: TBevel;
    tbtnCleanAll: TToolButton;
    tbtnMove: TToolButton;
    tbtnBrushEraser: TToolButton;
    tbtnColorize: TToolButton;
    procedure FormCreate(Sender: TObject);
    procedure tbtnOpenClick(Sender: TObject);
    procedure tbtnSwitchViewClick(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure tbtnSaveClick(Sender: TObject);
    procedure tbtnSaveAsClick(Sender: TObject);
    procedure pmiAboutClick(Sender: TObject);
    procedure sbtnZoomDownClick(Sender: TObject);
    procedure sbtnZoomUpClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure tbtnOptionsClick(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure i32ClrSelClick(Sender: TObject);
    procedure gOutMouseWheel(Sender: TObject; Shift: TShiftState;
      WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
    procedure i32ClrSelMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer; Layer: TCustomLayer);
    procedure gOutMouseLeave(Sender: TObject);
    procedure tbtnNightViewContextPopup(Sender: TObject; MousePos: TPoint;
      var Handled: Boolean);
    procedure i32ClrSelMouseEnter(Sender: TObject);
    procedure i32ClrSelMouseLeave(Sender: TObject);
    procedure tbtnCloseClick(Sender: TObject);
    procedure tbtnBackgroundClick(Sender: TObject);
    procedure pmiAboutBlankClick(Sender: TObject);
    procedure pmiAboutTestClick(Sender: TObject);
    procedure pmiHelpClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure gOutMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer; Layer: TCustomLayer);
    procedure tbtnFixBackgroundClick(Sender: TObject);
    procedure tbtnPenClick(Sender: TObject);
    procedure gOutMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer; Layer: TCustomLayer);
    procedure gOutMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer; Layer: TCustomLayer);
    procedure tbtnFloodFillClick(Sender: TObject);
    procedure cbFillDiagClick(Sender: TObject);
    procedure tbtnPxEraserClick(Sender: TObject);
    procedure CbFramecbNightClick(Sender: TObject);
    procedure CbFramecbPlayerClick(Sender: TObject);
    procedure tbtnFillEraserClick(Sender: TObject);
    procedure tbtnCleanAllClick(Sender: TObject);
    procedure tbtnMoveClick(Sender: TObject);
    procedure tbtnBrushEraserClick(Sender: TObject);
    procedure tbtnUndoClick(Sender: TObject);
    procedure tbtnRedoClick(Sender: TObject);
    procedure tbtnColorizeClick(Sender: TObject);
    //..........................................................................
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    // These two are changed to deal with private variables, it's more proper
    // than using OnCreate and OnDestroy for that.
  protected //------------------------------------------------------------------
    procedure WMSysCommand(var Message: TWMSysCommand); message WM_SYSCOMMAND;
    procedure WMEraseBkgnd(var Msg: TWMEraseBkgnd); message WM_ERASEBKGND;
    // the whole form is covered, no need for repainting background;
    // some (small) time can be saved instead ...
  private //--------------------------------------------------------------------
    FFileName: String;
    // name of opened file
    FPictureModified: Boolean;
    // changed?
    FEngine: TPictureEngine;
    // THE thing
    FOptions: TSdiEdOptions;
    // record for program options, see declaration in "OptionU"
    // very important part, too
    FTool: TToolBase;
    // currently used tool; take care to NIL when not used!
    FArrowNormal, FArrowHighlight: TBitmap32;
    // arrow graphic for color selector - are made on startup using system colors
    FclBtnFace32, FclBtnText32, FclHighlight32, FclHighlightText32: TColor32;
    // system colors in "our" format - for color selector, too
    FSelectedColor: TColor32;
    // current (selected) color for painting and such
    FOldGoutWndProc: TWndMethod;
    // old wnd proc of gOut
    FBackFiles: TStringList;
    // files used previously as backgrounds
    FMouseWheel: Boolean;
    // mouse has wheel
    FChunkList: TList;
    // list for storing chunks that are kept from old file
    FTrackSize: Boolean;
    // track form size changes
    FOldClrSelOverIndex: Integer;
    // previously used index for color selector
    bsbtnBrushSel: TBrushButton;
    function LoadFile(const AFileName: String): Boolean;
    // load routine
    procedure SaveImage(const AFileName: String);
    // save routine
    procedure LoadRes(const AName: String);
    // load resource
    procedure DoCaption(const ACaption: String);
    // set window caption
    procedure RefreshFromDisk;
    // update from the same file
    procedure IdealZoom(const Tight: Boolean = True);
    // set zoom so that the whole picture is visible
    procedure DownToolButton(const ToolButton: TToolButton);
    // set tool button as pressed and execute its event
    procedure DoZoom(const Value: Integer); overload;
    procedure DoZoom(const Value: Single); overload;
    // set zoom level and label
    procedure Zoom(const Ratio: Single);
    // change zoom level
    procedure ToggleFullScreen;
    // if maximized, restore; if not, maximize
    procedure TryPaste;
    // attempt at pasting from clipboard
    procedure CopyToClipboard;
    // copy picture to windows clipboard as bitmap
    procedure SetIndicator(const Position: TPoint);
    // set indicator in bottom left corner - color, pos, hex, highlight or outside
    procedure HandleDrop(var Msg: TWMDropFiles);
    // handle dropped file(s)
    procedure NewGoutWndProc(var Message: TMessage);
    // new gOut wnd proc
    procedure ConditionalBeep(const BeepType: Cardinal);
    // beep if options allow it
    function GetViewportMidpoint: TPoint;
    // returns coords of a point in the middle of gOut
    function CreateTool(const AToolType: TToolClass): TToolBase;
    // create a tool of given type
    procedure UseTool(const AToolType: TToolClass);
    // exchange current tool for new of given type
    procedure PictureChanged(Sender: TObject);
    // set status to changed, update undo buttons
    procedure BrushChanged(Sender: TObject);
  protected
    procedure SetOptions(const AOptions: TSdiEdOptions);
    // set internal options field and adjust things - used for property "Options"
  public //---------------------------------------------------------------------
    function AddBackFile(const FileName, Alias: String): Integer;
    procedure SortBackFilesByDisplayName;
    property Options: TSdiEdOptions read FOptions write SetOptions;
    // see procedure SetOptions ^^
    property BackFiles: TStringList read FBackFiles;
    property ViewportMidpoint: TPoint read GetViewportMidpoint;
    property SelectedColor: TColor32 read FSelectedColor;
    property Engine: TPictureEngine read FEngine;
  end;

//------------------------------------------------------------------------------

var MainFrm: TMainFrm;
    AppPath, DataPath: String;
    // AppPath: what folder are we in, initialized in the beginning by unit
    // DataPath: what folder is safe for user data

const cOpFile = 'Options.ini';
      cBkFile = 'Backgrounds.ini';
      // for handling options ini file
      ErrorBeep: Cardinal = MB_ICONHAND;
      QuestionBeep: Cardinal = MB_ICONASTERISK;
      // MessageBeep constants for easier changing (sound aesthetics ... lol)

//==============================================================================

implementation

uses Types, Clipbrd, IniFiles, Registry, Math, ShellAPI, StrUtils, TypInfo,
     // Delphi
     GR32_Blend, PNGImage,
     // additional
     ToolBkFixU, ToolPaintPencilU, ToolFillU, ToolMagicPencilU, ToolMagicFillU,
     ToolMoverU, ToolCleanAllU, ToolMagicBrushU, ToolColorizeU,
     // this project - tools
     ConstStrU, SysLowU, CoreLowU, ShadesInOutU, CalcUtilU, BackgroundEngineU,
     // this project - nonvisual
     ColorU, StrU, FileU, TranslationU, TransGuiU,
     // other custom
     AboutU, StatU, ClChgU, NightOptU, BackPicU, ModuleU, Buttons;
     // this project - gui

{$R *.dfm}

//------------------------------------------------------------------------------

function CompareBackFiles(List: TStringList; Index1, Index2: Integer): Integer;
begin
  Result:= AnsiCompareText(List.ValueFromIndex[Index1], List.ValueFromIndex[Index2]);
end;

//==============================================================================

procedure TMainFrm.FormCreate(Sender: TObject);
var S, ToOpen: String;
    Ini: TCustomIniFile;
    Translations: TStrings;
begin
  bsbtnBrushSel:= TBrushButton.Create(pnlTools);
  with bsbtnBrushSel do begin
    OnBrushSelect:= BrushChanged;
    Left:= CbFrame.Left + CbFrame.Width;
    Top:= 0;
    Visible:= True;
    Parent:= pnlTools;
    Hint:= 'Brush selection (R)'; //t+ // will get translated with whole form
    ShowHint:= True;
  end;

  FOptions.IniFileKind:= iftFile;
  // default is file
  case ParamCount of
    1: ToOpen:= ParamStr(1);
    2: if UpperCase(ParamStr(1)) = '-DATA' then begin
      DataPath:= ParamStr(2);
    end;
    3: if UpperCase(ParamStr(1)) = '-DATA' then begin
      DataPath:= ParamStr(2);
      ToOpen:= ParamStr(3);
    end;
  end;
  // handle parameters
  if DataPath = EmptyStr then begin // no specified path? use default
    DataPath:= GoodSlashes(GetAppDataPath + UserDataDirPostfix);
    // data path is now known, check availability and crash if not
    if not ForceDirTree(DataPath) then begin
      DataPath:= GoodSlashes(AppPath + 'Shades_5.0_data/');
      if not ForceDirTree(DataPath) then begin
        MessageBeep(ErrorBeep);
        Application.MessageBox(PChar(_('Unable to use designed data path!')), //t+
          PChar(Translate(sscMsgCapError)),
          MB_OK + MB_ICONERROR);
        raise EInOutError.Create(_('Unable to use designed data path!')); //t+
      end;
    end;
  end else begin // path WAS specified
    DataPath:= GoodSlashes(DataPath);
    if DataPath[1] = '@' then begin // is it special?
      S:= UpperCase(DataPath);
      if (S = '@REGISTRY') or (S = '@REG') then begin
        FOptions.IniFileKind:= iftRegistry;
        DataPath:= 'Software\Shades\5.0\';
      end;
      if S = '@NONE' then FOptions.IniFileKind:= iftNone;
    end else begin // only normal path
      if DataPath[1] = PathDelim then Delete(DataPath, 1, 1);
      // remove first char if it is \ or /
      if DataPath[Length(DataPath)] <> PathDelim then
        DataPath:= DataPath + PathDelim;
      // add \ or / as last if needed
      if (Length(DataPath) < 3) or (DataPath[2] <> DriveDelim) then
        DataPath:= AppPath + DataPath;
      // relative path? append to executable path
    end;
  end;

  // end of path detections and corrections
  case FOptions.IniFileKind of
    iftFile: Ini:= TIniFile.Create(DataPath + cOpFile);
    iftRegistry: Ini:= TRegistryIniFile.Create(DataPath + cOpFile);
    iftNone: Ini:= nil;
  end;
  // handle INI kind
  if not ((FOptions.IniFileKind = iftNone) or (Ini = nil)) then try
    // loading options from file or registry
    // TODO: see if the check for nil is needed
    LoadOptions(Ini, FOptions);
    LoadBackgrounds(GoodSlashes(DataPath + cBkFile), FBackFiles, FOptions.IniFileKind);
    SortBackFilesByDisplayName;
    SetOptions(FOptions); // apply options
  finally
    Ini.Free;
  end else begin // do not read anything
    LoadDefaultOptions(FOptions);
    SetOptions(FOptions); // apply options
  end;
  FMouseWheel:= Mouse.WheelPresent;
  with FOptions do begin
    Self.Top:= Window.Top;
    Self.Left:= Window.Left;
    Self.Height:= Window.Height;
    Self.Width:= Window.Width;
    Self.WindowState:= Window.State;
  end;

  FOldGoutWndProc:= gOut.WindowProc;
  gOut.WindowProc:= NewGoutWndProc;
  DragAcceptFiles(gOut.Handle, True);
  // drag & drop support for gOut - replace wnd proc and keep pointer to old one

  FEngine.Options:= FOptions; // WATCH OUT FOR THIS !!!!
  FEngine.ViewMode:= vmNormal;
  FEngine.Viewport:= gOut;
  FEngine.OnChange:= Self.PictureChanged;
  FEngine.ToolOptions.LoadFromForm(Self);
  FEngine.ToolOptions.Brush:= bsbtnBrushSel.SelectedBrush;

  i32ColorView.Bitmap.SetSize(12, 12);
  i32ColorView.Top:= StatusBar.Top + 4;
  // Sometimes mysteriously "runs away", so make sure it will be where wanted.
  lbPosData.Caption:= EmptyStr;
  // initialize some parts of GUI

  Caption:= sscStartAppCap; // keep as const, easier management
  if ToOpen <> EmptyStr then begin // open?
    FFileName:= GoodSlashes(ToOpen);
    if not LoadFile(FFileName) then begin
      // if that file does not work, scream and load default picture
      LoadRes('DefaultImage');
      FFileName:= EmptyStr;
      S:= Format(_('File %s could not be opened.'), [FFileName]); //tf
      MessageBeep(ErrorBeep);
      Application.MessageBox(PChar(S), PChar(Translate(sscMsgCapError)),
        MB_OK + MB_ICONERROR);
    end else begin
      DoCaption(ExtractFileName(FFileName));
    end;
  end else begin
    LoadRes('DefaultImage');
    FFileName:= EmptyStr;
  end;
  // or if there was no parameter, do that ^ too
  if FOptions.FirstRun then try
    // if this is new installation, try and determine language automatically
    Translations:= TStringList.Create;
    FindFilesByMask(GoodSlashes(AppPath + 'translations/*.po'), Translations);
    // first load list of what we have to choose from
    FOptions.InterfaceOptions.LangFile:= ChooseTranslationFile(
      GetSystemLanguageName, Translations, 'en.po');
    // then pass
  finally
    Translations.Free;
  end;
  // now language is set, load it
  with FOptions.InterfaceOptions do if LangFile <> EmptyStr then begin
    S:= GoodSlashes(AppPath + 'translations/' + LangFile);
    if FileExists(S) then ChangeTranslation(S);
  end;
  // and use
  TranslateComponent(Self);
  Font.Name:= Translate(sscFontName);

  with i32ClrSel do Bitmap.SetSize(Width, Height); // buffer must be set at runtime
  i32ClrSelClick(Sender); // force the first repaint

  DoZoom(FOptions.DefZoomLevel);
  DownToolButton(tbtnPen);
end;

//------------------------------------------------------------------------------

procedure TMainFrm.FormDestroy(Sender: TObject);
var Ini: TCustomIniFile;
begin
  // when closing program, save options
  case FOptions.IniFileKind of
    iftFile: Ini:= TIniFile.Create(DataPath + cOpFile);
    iftRegistry: Ini:= TRegistryIniFile.Create(DataPath + cOpFile);
    iftNone: Ini:= nil;
  end;
  // handle kind of ini
  if not ((FOptions.IniFileKind = iftNone) or (Ini = nil)) then begin
    if FOptions.IniFileKind = iftFile then ForceDirTree(DataPath);
    with Ini, FOptions do try
      if Window.State <> wsMaximized then begin
        Window.Top:= Self.Top;
        Window.Left:= Self.Left;
        Window.Height:= Self.Height;
        Window.Width:= Self.Width;
      end;
      FOptions.FirstRun:= False;
      SaveOptions(Ini, FOptions);
      SaveBackgrounds(DataPath + cBkFile, FBackFiles, IniFileKind);
    finally
      UpdateFile;
      Ini.Free;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
(* Handler for key presses; most of keys do call buttons' handlers, anyway.
   Since the CASEs inside this procedure body got really lenghty, I split the
   in-place code into private procedures of form class. The new goal is to make
   all case coding just one statement. *)
begin
  if [ssShift, ssCtrl] = Shift then case Key of // with SHIFT and CTRL
    Ord('S'): tbtnSaveAs.Click;
  end else if ssCtrl in Shift then case Key of // with CTRL
    // if ever used, ctrl-alt keys must go before this!
    Ord('A'): tbtnCleanAll.Click;
    Ord('B'): tbtnFixBackground.Click;
    Ord('C'): CopyToClipboard;
    Ord('O'): tbtnOpen.Click;
    Ord('S'): tbtnSave.Click;
    Ord('V'): TryPaste;
    Ord('Y'), Ord('R'): tbtnRedo.Click;
    Ord('Z'): tbtnUndo.Click;
  end else if ssAlt in Shift then case Key of // with ALT
    Ord('O'): tbtnOptions.Click;
    VK_RETURN: ToggleFullScreen;
    VK_BACK: tbtnUndo.Click;
  end else case Key of // with no modifier keys
    Ord('B'): tbtnBackground.Click;
    Ord('D'): DownToolButton(tbtnDayView);
    Ord('N'): DownToolButton(tbtnNightView);
    Ord('S'): DownToolButton(tbtnSpecView);
    Ord('1'): DownToolButton(tbtnPen);
    Ord('2'): DownToolButton(tbtnFloodFill);
    Ord('3'): DownToolButton(tbtnPxEraser);
    Ord('4'): DownToolButton(tbtnBrushEraser);
    Ord('5'): DownToolButton(tbtnFillEraser);
    Ord('6'): DownToolButton(tbtnMove);
    Ord('R'): bsbtnBrushSel.Click;
    VK_ADD: sbtnZoomUpClick(Sender);
    VK_SUBTRACT: sbtnZoomDownClick(Sender);
    VK_NUMPAD0: DoZoom(FOptions.DefZoomLevel); // defult zoom level
    VK_MULTIPLY: IdealZoom;
    VK_DIVIDE: IdealZoom(False);
    VK_HOME: gOut.ScrollToCenter(0, 0);
    VK_END: with gOut, Bitmap do ScrollToCenter(Width - 1, Height - 1);
    // zooming stuff
    VK_F1: pmiHelp.Click;
    VK_F5: RefreshFromDisk;
    VK_ESCAPE: if Assigned(FTool) then FTool.Cancel;
  end;
end;

//------------------------------------------------------------------------------

function TMainFrm.LoadFile(const AFileName: String): Boolean;
// - loads picture from file; handles exceptions and returns true if successful
// - in some cases, areas may be left (refresh)
var P: TPngObject;
begin
  Result:= True; // assume it worked
  P:= TPNGObject.Create;
  try
    try
      if (AFileName = EmptyStr) or not FileExists(AFileName) then
        raise Exception.CreateFmt(_('File %s does not exist.'), [AFileName]); //tf
      // handle non-existent file
      P.LoadFromFile(AFileName);
      with P do if Empty or (Width * Height = 0) then
        raise Exception.Create(_('Image is empty!')); //t+
      (* If there is nothing -> :( Two conditions, because "TPNGObject.Empty" looks
         for number of PNG chunks, not picture size. Both are cause for concern here.
         Here is the last point where exceptions may break flow of statements, so
         the following calls are executed only if loading was succesful.*)
      FEngine.Picture.Assign(P);
      // Works because TPNGObject is descendant of TGraphic. Reversed way impossible!
      ClearChunkList(FChunkList);
      SaveChunks(P, FChunkList);
      AddToRecent(AFileName);
      FEngine.Reloaded;
      // reload everything
    except
      on Exception do Result:= False; // if something didn't work
    end;
  finally
    P.Free;
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.tbtnOpenClick(Sender: TObject);
  //............................................................................
  procedure DoOpen;
  (* This checks for special strings before loading file and then takes care of
    other things. *)
  var UpFileName: String;
      abPos: Integer;
  begin
    with DialogModule.OpenPictureDialog do if Execute then begin
      FFileName:= FileName;
      UpFileName:= UpperCase(FFileName);
      abPos:= Pos('ABOUT:', UpFileName);
      if abPos <> 0 then
        UpFileName:= Copy(UpFileName, abPos, Length(UpFileName) - abPos + 1);
      (* Search full file name (drive + path + name + extension) for any
         "about:". If found, it is no real file but means of telling the program
         to load its resource pictures. Under Windows, no file name may contain
         character ":", so this does not hide any files beginning with "about:"
         as they cannot exist. If there was "ABOUT:", retrieve it with its
         parameter and forget everything else - when you write something into
         OpenDialog box and press open, the text is appended to current folder,
         but we want to have just the part with "about:".
      *)
      if UpFileName = 'ABOUT:BLANK' then begin
        LoadRes('DefaultImage'); // the picture of crane (startup one)
        FFileName:= EmptyStr;
        DoCaption('About:Blank');
      end else if UpFileName = 'ABOUT:TEST' then begin
        LoadRes('TestImage'); // picture with all special colors, useful for testing
        FFileName:= EmptyStr;
        DoCaption('About:Test');
      end else if UpFileName = 'ABOUT:EASTEREGG' then begin
        gOut.Scale:= 1;
        LoadRes('EasterEgg'); // why not :-)
        FFileName:= EmptyStr;
        DoCaption('About:EasterEgg');
      end else if LoadFile(FFileName) then begin
        (* ... it wasn't any special filename it did not fit any special string,
          treat as file name and if OK adjust stuff *)
        FPictureModified:= False;
        if FFileName <> EmptyStr then DoCaption(ExtractFileName(FFileName))
        else DoCaption(_('<no file>')); //t+
      end;
    end;
  end;
  //............................................................................
begin // HERE begins the tbtnOpenClick itself !!!
  if not FPictureModified then begin // nothing to do before reloading
    DoOpen;
  end else begin // unsaved changes, ask user
    //ConditionalBeep(QuestionBeep);
    case Application.MessageBox(
        PChar(_('Picture is not saved. Do you want to save it now?')), //t+
        PChar(Translate(sscMsgCapProceed)),
        MB_YESNOCANCEL + MB_ICONQUESTION) of
      IDYES: begin
        tbtnSaveClick(Sender);
        // Calling the procedure for clicking save button is easier than
        // writing it all again here.
        DoOpen; // and now open the new pic with ^^^
      end;
      IDNO: DoOpen;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.tbtnSwitchViewClick(Sender: TObject);
// universal for all 3 buttons, these have an integer value in Tag
begin
  FEngine.ViewMode:= (Sender as TComponent).Tag; // set number from button
end;

//------------------------------------------------------------------------------

procedure TMainFrm.SaveImage(const AFileName: String);
var P: TPNGObject;
    B: TBitmap;
begin
  // save to PNG
  P:= TPNGObject.Create;
  B:= TBitmap.Create;
  try
    (* TBitmap32 has some smart support for all TGraphic descendants, but
       TPNGObject implements only assigning from its own type and TBitmap.
       I have to move the picture with help of additional instance of TBitmap,
       which both involved classes support. *)
    B.Assign(FEngine.Picture);
    P.Assign(B);
    P.CompressionLevel:= FOptions.PNGCompression;
    LoadChunks(P, FChunkList);
    if FOptions.TimeStamp then AddTimestamp(P);
    P.SaveToFile(AFileName);
    DoCaption(ExtractFileName(AFileName));
  finally
    P.Free;
    B.Free;
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.tbtnSaveClick(Sender: TObject);
begin
  if FPictureModified then begin
    if FFileName = EmptyStr then begin
      tbtnSaveAsClick(Sender);
      // if it is not a file, just do "save as"
    end else begin
      SaveImage(FFileName);
      FPictureModified:= False;
      // changes were saved
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.tbtnSaveAsClick(Sender: TObject);
begin
  with ModuleU.DialogModule.SavePictureDialog do if Execute then begin
    FFileName:= FileName;
    SaveImage(FFileName);
    FPictureModified:= False;
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.pmiAboutClick(Sender: TObject);
begin
  AboutFrm.ShowModal;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.Zoom(const Ratio: Single);
var NewScale: Single;
    MidPoint: TPoint;
begin
  with gOut do begin
    NewScale:= Scale * Ratio;
    if InRange(NewScale, 1, 32) then begin
      BeginUpdate;
      MidPoint:= ControlToBitmap(ViewportMidpoint);
      Scale:= NewScale;
      with MidPoint do ScrollToCenter(X, Y);
      lbZoom.Caption:= Format(_('Zoom: %dx'), [Round(Scale)]); //tf
      EndUpdate;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.sbtnZoomDownClick(Sender: TObject);
begin
  Zoom(1/2);
end;

//------------------------------------------------------------------------------

procedure TMainFrm.sbtnZoomUpClick(Sender: TObject);
begin
  Zoom(2);
end;

//------------------------------------------------------------------------------

procedure TMainFrm.SetOptions(const AOptions: TSdiEdOptions);
begin
  tbtnClose.Visible:= AOptions.InterfaceOptions.ShowCloseBtn;
  with gOut do if Scale = FOptions.DefZoomLevel then Scale:= AOptions.DefZoomLevel;
  // if zoom was default, change it to new default
  if AOptions.MouseOpt.CustomCursors then gOut.Cursor:= crCrossDef else
    gOut.Cursor:= crCross;
  pnlTools.Align:= AOptions.InterfaceOptions.ToolBarAlign;
  FOptions:= AOptions;
  FEngine.Options:= AOptions;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.tbtnOptionsClick(Sender: TObject);
// Offer dialog with options and if some are changed, apply.
begin
  with OptionFrm do begin
    Options:= FOptions;
    // OptionFrm.Options := MainFrm.FOptions
    if ShowModal = mrOk then Self.Options:= Options;
    // Self = MainFrm => MainFrm.Options := OptionFrm.Options
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  if FPictureModified then begin
    //ConditionalBeep(QuestionBeep);
    case Application.MessageBox(
        PChar(_('Do you want to save file before exit?')), //t+
        PChar(Translate(sscMsgCapClose)),
        MB_YESNOCANCEL + MB_ICONQUESTION) of
      IDYES: begin
        if FFileName = EmptyStr then tbtnSaveAsClick(Sender) else
          SaveImage(FFileName);
        CanClose:= True;
      end;
      IDNO: CanClose:= True;
      IDCANCEL: CanClose:= False;
    end;
  end else begin
    CanClose:= True;
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.LoadRes(const AName: String);
var P: TPNGObject;
begin
  P:= TPNGObject.Create;
  try
    P.LoadFromResourceName(hInstance, AName);

    FEngine.Picture.Assign(P);
    FEngine.Reloaded;

    FPictureModified:= False;
  finally
    P.Free;
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.DoCaption(const ACAption: String);
begin
  Caption:= Translate(sscAppCap) + ACaption;
  tbtnSave.Enabled:= not ((Copy(ACaption, 1, 6) = 'About:') or
    (ACaption = _('<no file>')) or //t+
    (ACaption = _('<from clipboard>'))); //t+
end;

//------------------------------------------------------------------------------

procedure TMainFrm.RefreshFromDisk;
(* Try reloading the image from disk to see changes from external editing or sth.
  else or revert unwanted changes since last save without hitting undo like mad
  if it was pixel-pushing. *)
var BackupBuf: TBitmap32;
  //...................................................
  procedure FallBack;
  begin
    (* WARNING - LoadFile called as first thing (see below) calls ReloadImages
      which updates the viewport with loaded picture. Every time I want to
      revert changes made by loading, I must also force one more reload; if I
      wouldn't, reverting reload would take place only internally and would
      look like true reload. *)
    FEngine.Picture.Assign(BackupBuf); // revert loading picture
    FEngine.Reloaded;
  end;
  //...................................................
begin
  BackupBuf:= TBitmap32.Create;
  gOut.BeginUpdate;
  (* The code is slightly confusing because I do not resolve cases when I
    should/can refresh (load from disk) as would be guessed from the user's
    logic point of view. Everything works exactly vice versa.
    First thing to do is making a backup of current picture in BackupBuf, then I
    try to load. Then, depending on errors or user's decisions, I revert
    loading by using backup. So, the whole logic is inverted - loading is a
    given thing, then lots of IF-ELSEs that "undo" that action. *)
  if FFileName = EmptyStr then raise EAbort.Create(_('Refresh: not a file!')); //t+
  (* Exit silently if the picture is not associated with any filename.
     Assumption is that window caption already tells this to user and he
     remembers his action that imported the picture. *)
  if FileExists(FFileName) then begin // is it still there?
    BackupBuf.Assign(FEngine.Picture);
    (* Make a backup of what we have now so that work won't be lost even if some
      actions fail. *)
    if not LoadFile(FFileName) then begin
      (* Open the file again, and if failed, ask what to do. We know the file
        exists. *)
      //ConditionalBeep(ErrorBeep);
      if Application.MessageBox(PChar(Translate(sscReloadInvalid)), // too long
          PChar(Translate(sscMsgCapError)), MB_YESNO + MB_ICONWARNING)
        = IDYES then tbtnSave.Click;
      FallBack;
      // fallback after unsuccesful reload
    end else begin // everything seems OK, but did the user make any changes?
      if FPictureModified then begin
        // conflict - picture changed, ask
        //ConditionalBeep(QuestionBeep);
        if (Application.MessageBox(PChar(Translate(sscReloadWarning)), // too long
            PChar(_('Reload?')), MB_ICONQUESTION + MB_YESNO ) //t+
          = IDNO) then FallBack
        else FPictureModified:= False;
        (* Reload = already done, and now any changes will be surely lost, so
          set proper status. *)
      end; // FPictureModified: ELSE -> do not revert = do nothing
    end;
  end else begin // when the file disappeared since opening
    //ConditionalBeep(ErrorBeep);
    Application.MessageBox(PChar(Translate(sscReloadNotExists)), // too long
      PChar(Translate(sscMsgCapError)), MB_OK + MB_ICONERROR);
  end;
  gOut.EndUpdate;
  gOut.Refresh;
  BackupBuf.Free;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.i32ClrSelClick(Sender: TObject);
// repaint color selector and change selected color
var NewIndex, i, j: Integer;
    Rct: TRect;
begin
  NewIndex:= EnsureRange((i32ClrSel.ScreenToClient(Mouse.CursorPos).X - 1) div 17 + 1,
    ScaAllStart, ScaAllEnd);
  (* Crop values - this routine is called at startup and the mouse position is
     then NOT over the control -> index out of range. *)
  with i32ClrSel do begin
    BeginUpdate;
    with Bitmap do begin
      Clear(FclBtnFace32); // everything has now "buttonish" color
      RaiseRectTS(0, 0, Width, Height, -100); // lowered bevel around control
      for i:= ScaAllStart to ScaAllEnd do with rct do begin
        j:= (i - 1) * 17 + 1;
        // Uhh, I can't remember what this "j" exactly is, but it helps in
        // further math.
        if i = NewIndex then begin // is this the selected one?
          Top:= 1;
          Bottom:= 29;
          Left:= j;
          Right:= j + 17; // these are rct.?, not from control !
          FillRect(Left, Top, Right, Bottom, FclHighlight32);
          // change background of this one to make it look selected (highlighted)
          FSelectedColor:= SpecColorArray[i].Normal;
          (* This is the wanted color, put it aside - I save only color, not its
             index, it is enough now. *)
        end;
        Top:= 3;
        Left:= j + 2;
        Bottom:= 12;
        Right:= j + 15;
        FillRectS(rct, SpecColorArray[i].Normal);
        // paint upper part - original color
        if i = NewIndex then FrameRectS(rct, FclHighlightText32) else
          FrameRectS(rct, FclBtnText32);
        // rectangle around color - normal or selected
        Top:= 19;
        Bottom:= 27;
        FillRectS(rct, SpecColorArray[i].Special);
        // lower part - dark look
        if i = NewIndex then FrameRectS(rct, FclHighlightText32) else
          FrameRectS(rct, FclBtnText32);
        // again rectangle around color
        DrawMode:= dmBlend;
        if i = NewIndex then Draw(j + 7, 13, FArrowHighlight) else
          Draw(j + 7, 13, FArrowNormal);
        // arrow graphic
      end;
    end;
    EndUpdate;
    Repaint;
    lbEditData.Caption:= IfThen(
      FOptions.InterfaceOptions.DecimalIndicator,
      Format(_('Selected (%s)'), [Cl32ToDecPlain(FSelectedColor)]), //tf
      Format(_('Selected $%s'), [Cl32ToHexPlain(FSelectedColor)]) ); //tf
    FEngine.ToolOptions.Color:= FSelectedColor;
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.gOutMouseWheel(Sender: TObject; Shift: TShiftState;
  WheelDelta: Integer; MousePos: TPoint; var Handled: Boolean);
var PosScrollShift: Boolean;
begin
  with FOptions.MouseOpt, gOut do begin
    PosScrollShift:= HorizontalScroll;
    if PosScrollShift then
      PosScrollShift:= (ScreenToClient(MousePos).Y / Height * 100) >
      (100 - HorizontalPart);
    if (ssShift in Shift) or PosScrollShift then
      // with shift (or lower than set) horizontal
      Scroll(WheelDelta * ScrollFactorY div -100, 0)
    else // normally vertical
      Scroll(0, WheelDelta * ScrollFactorX div -100);
    Handled:= True;
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.IdealZoom(const Tight: Boolean = True);
var xRatio, yRatio, usedRatio: Real;
    i: Integer;
begin
  (* Zoom in as much as possible - using the same zoom levels as user can! Arrows
     do not set proper values, just multiply and divide by 2 -> mustn't use any
     other zoom level than 1,2,4,8,16,32. *)
  with gOut do begin
    xRatio:= Width / Bitmap.Width;
    yRatio:= Height / Bitmap.Height;
  end;
  usedRatio:= IfThen(Tight, Min(xRatio, yRatio), Max(xRatio, yRatio));
  // pick either the more or less restricting ratio out of both axes' ones
  for i:= 0 to 5 do if ( (1 shl i) > usedRatio ) then Break;
  (* Iterate through all possible zoom levels - 2^i done as bit shift; if
     picture does not fit, stop the cycle -> "i" keeps the last used value. *)
  if i > 0 then dec(i);
  (* Cycle stopped with first too big value -> lower it for proper result, but
     if it is 0, leave it as is because zoom must be >= 1. Now we have in "i"
     the last zoom level that allows the whole picture to be displayed. *)
  DoZoom( 1 shl i );
  // So let's use it - remember it must be again converted to power of 2.
end;

//------------------------------------------------------------------------------

procedure TMainFrm.DownToolButton(const ToolButton: TToolButton);
// helper routine to set TToolButton as pressed - 2 in 1
begin
  with ToolButton do begin
    Down:= True;
    Click;
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.DoZoom(const Value: Integer); // OVERLOAD
// set zoom and take care of label - 2 in 1
begin
  gOut.Scale:= Value;
  lbZoom.Caption:= Format(_('Zoom: %dx'), [Value]); //tf
end;

procedure TMainFrm.DoZoom(const Value: Single); // OVERLOAD
begin
  gOut.Scale:= Value;
  lbZoom.Caption:= Format(_('Zoom: %sx'), //tf
    [FloatToStrF(Value, ffGeneral, 3, 2)]);
end;

//------------------------------------------------------------------------------

procedure TMainFrm.i32ClrSelMouseMove(Sender: TObject; Shift: TShiftState;
  X, Y: Integer; Layer: TCustomLayer);
var OverIndex: Integer;
begin
  with memHint do begin
    OverIndex:= EnsureRange((X - 1) div 17 + 1, ScaAllStart, ScaAllEnd);
    if OverIndex <> FOldClrSelOverIndex then begin
      // finding the color under cursor, same as OnClick for this control
      Lines[0]:= _('Color selector: ') + IfThen( //t+
        FOptions.InterfaceOptions.DecimalIndicator,
        '(' + Cl32ToDecPlain(SpecColorArray[OverIndex].Normal) + ')',  // decimal
        Cl32ToHexPlain(SpecColorArray[OverIndex].Normal)); // hex (hax? :D)
      (* Order of colors is same as in SpecColorArray, since it is used to build
         the control, thus OverIndex can be directly used. *)
      Lines[1]:= Translate(SpecColorArray[OverIndex].Description);
      Perform(EM_SCROLL, SB_LINEUP, 0);
    end;
    FOldClrSelOverIndex:= OverIndex;
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.ToggleFullScreen;
// toggle maximized state
begin
  case WindowState of
    wsNormal: WindowState:= wsMaximized;
    wsMaximized: WindowState:= wsNormal;
  end;
  (* Better than IF, because this leaves the last possibility (wsMinimized)
     completely out of game in case some glitch occured. *)
end;

//------------------------------------------------------------------------------

procedure TMainFrm.TryPaste;
// handle pasting tries and all the involved user interaction
var B: TBitmap;
  //............................................................................
  procedure DoPaste;
  // paste picture from clipboard
  begin
    B:= TBitmap.Create;
    B.Assign(Clipboard);
    FEngine.Picture.Assign(B);
    (* TBitmap32 knows only itself, TBitmap and TGraphic children, but
       TClipboard is only ancestor of TPersistent. "Swapping" bitmap needed. *)
    FEngine.Reloaded;
    B.Free;
    FFileName:= EmptyStr;
    DoCaption(_('<from clipboard>')); //t+
    // begin with new picture
    FPictureModified:= True; // picture comes from clipboard so it wasn't saved
  end;
  //............................................................................
begin
  if FPictureModified then begin // paste from clipboard, ask user if needed
    if Clipboard.HasFormat(CF_PICTURE) then begin
      //ConditionalBeep(QuestionBeep);
      case Application.MessageBox(
          PChar(Translate(sscPasteQuestion)), PChar(Translate(sscMsgCapProceed)),
          MB_YESNOCANCEL + MB_ICONQUESTION)
      of
        IDYES: begin
          tbtnSaveClick(Self); // first save
          DoPaste; // then save
        end;
        IDNO: DoPaste; // do not save
        IDCANCEL: ; // do nothing
      end;
    end;
  end else begin
    if Clipboard.HasFormat(CF_PICTURE) then DoPaste;
    // or paste without asking if the previous picture was unmodified
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.CopyToClipboard;
var B: TBitmap;
begin
  B:= TBitmap.Create;
  try
    B.Assign(FEngine.Picture);
    Clipboard.Assign(B);
    // TClipboard does not know about TBitmap32, so "transition" bitmap is needed.
  finally
    B.Free;
  end;
end;


//------------------------------------------------------------------------------

procedure TMainFrm.SetIndicator(const Position: TPoint);
// set indicator - color or no info
var clr: TColor32;
    PositionStr: String;
begin
  if PtInRect(FEngine.BoundsRect, Position) then begin
    // if it is inside picture
    with Position do clr:= FEngine.Picture.PixelS[X, Y]
      {$IFNDEF DEBUG} or $FF000000 {$ENDIF};
    with i32ColorView.Bitmap do begin
      FrameRectS(0, 0, 12, 12, FclBtnText32);
      FillRect(1, 1, 11, 11, clr);
      try
        if IsBackgroundColor(clr) then begin
          lbPosData.Font.Color:= clGreen;
          Abort;
        end;
        if IsPlayerPriColor(clr) then begin
          lbPosData.Font.Color:= WinColor(FOptions.HighlightMode.PlayerPriHlC);
          Abort;
        end;
        if IsPlayerSecColor(clr) then begin
          lbPosData.Font.Color:= WinColor(FOptions.HighlightMode.PlayerSecHlC);
          Abort;
        end;
        if IsDarkColor(clr) then begin
          lbPosData.Font.Color:= WinColor(FOptions.HighlightMode.DarkHlC);
          Abort;
        end;
        lbPosData.Font.Color:= clWindowText;
      except
        // catch ABORT
        // lightly optimized for %# of occurences by order of checks
      end;
    end;
    with Position do begin
      if not FOptions.InterfaceOptions.DecimalIndicator then begin
        {$IFNDEF DEBUG}
        PositionStr:= Format('%d,%d : %s', [X, Y, Cl32ToHexShort(clr)]); //t-
        {$ELSE}
        PositionStr:= Format('%d,%d : %s/%s', [X, Y, Cl32ToHex(clr), //t-
          Cl32ToHex(gOut.Bitmap.PixelS[X, Y])]);
          // show also alpha values and viewport actual colour
        {$ENDIF}
        // hexadecimal color
      end else begin
        PositionStr:= Format('%d,%d : (%s)', [X, Y, Cl32ToDecPlain(clr)]); //t-
        // decimal color
      end;
    end;
    lbPosData.Caption:= PositionStr;
  end else with i32ColorView.Bitmap do begin // if outside...
    FrameRectS(0, 0, 12 ,12, FclBtnText32);
    FillRect(1, 1, 11, 11, clWhite32);
    Line(10, 1, 1, 10, clBlack32, True);
    Line(5, 1, 1, 5, clBlack32, True);
    Line(10, 6, 6, 10, clBlack32, True);
    // ... paint diagonal lines
    lbPosData.Font.Color:= clWindowText;
    lbPosData.Caption:= _('Outside picture'); //t+
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.gOutMouseLeave(Sender: TObject);
begin
  SetIndicator(OuterPoint);
  if Assigned(FTool) and (FTool is TMouseToolBase) then
    TMouseToolBase(FTool).MouseLeave;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.tbtnNightViewContextPopup(Sender: TObject;
  MousePos: TPoint; var Handled: Boolean);
var Old: Cardinal;
begin
  with FOptions.DarkMode do begin
    Old:= Weight;
    Weight:= NightOptFrm.GetInput(Weight);
    if Weight <> Old then
      FEngine.Options:= FOptions;
  end;
  Handled:= True;
end;     

//------------------------------------------------------------------------------

procedure TMainFrm.i32ClrSelMouseEnter(Sender: TObject);
begin
  with memHint, Lines do begin
    Show;
    Clear;
    Append(EmptyStr);
    Append(EmptyStr);
  end;
  FOldClrSelOverIndex:= -1;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.i32ClrSelMouseLeave(Sender: TObject);
begin
  memHint.Hide;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.tbtnCloseClick(Sender: TObject);
begin
  Close;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.HandleDrop(var Msg: TWMDropFiles);
var P: PChar;
    S: String;
    FileCount, FileStrSize: Integer;
begin
  P:= nil;
  FileCount:= DragQueryFile(Msg.Drop, $FFFFFFFF, P, 0); // last param 255?
  case FileCount of // how many of them?
    0, 2..MaxInt: begin
      //ConditionalBeep(ErrorBeep);
      Application.MessageBox(
        PChar(_('You must drop exactly one picture file!')), //t+
        PChar(Translate(sscMsgCapError)), MB_OK + MB_ICONERROR);
    end;
    1: begin // only one file - that's what we really want
      FileStrSize := DragQueryFile(Msg.Drop, 0, nil, 0) + 1;
      P:= StrAlloc(FileStrSize);
      DragQueryFile(Msg.Drop, 0, P, FileStrSize);
      S:= String(P);
      // some API magic mixed with NTS ;-)
      if FileExists(S) then begin
        if UpperCase(ExtractFileExt(S)) = '.PNG' then begin
          if FPictureModified then begin
            //ConditionalBeep(QuestionBeep);
            case Application.MessageBox(
              PChar(_('Picture is not saved. Do you want to save it now?')), //t+
              PChar(Translate(sscMsgCapProceed)),
              MB_YESNOCANCEL + MB_ICONQUESTION)
            of
              IDYES: begin
                tbtnSave.Click;
                LoadFile(S);
                DoCaption(ExtractFileName(S));
                PictureChanged(Self);
                FFileName:= S;
              end;
              IDNO: begin
                LoadFile(S);
                DoCaption(ExtractFileName(S));
                PictureChanged(Self);
                FFileName:= S;
              end;
            end;
          end else begin
            LoadFile(S);
            DoCaption(ExtractFileName(S));
            PictureChanged(Self);
            FFileName:= S;
          end;
        end else begin
          //ConditionalBeep(ErrorBeep);
          Application.MessageBox(
            PChar(_('The dropped file is not a PNG picture!')), //t+
            PChar(Translate(sscMsgCapError)), MB_OK + MB_ICONEXCLAMATION);
          // if it wasn't png -> :(
        end;
      end; // if the file did not exist ... stay silent
    end;
  end;
  //StrDispose(P);
  (* Remove PChar properly; unfortunately, the memory of P is
     taken over by S after typecast -> done automatically (as of D7 Pers) *)
  DragFinish(Msg.Drop);
end;

//------------------------------------------------------------------------------

procedure TMainFrm.NewGoutWndProc(var Message: TMessage);
begin
 if Message.Msg = WM_DROPFILES then HandleDrop(TWMDropFiles(Message));
 FOldGoutWndProc(Message);
end;

//------------------------------------------------------------------------------

procedure TMainFrm.tbtnBackgroundClick(Sender: TObject);
var BS: TBackgroundSetting;
    C: TColor32;
begin
  if Assigned(BackPicFrm) then with BackPicFrm do begin
    lbPictures.Items.Clear;
    ReloadFileList;
    if ShowModal = mrOk then begin
      BS:= SelectedState;
      case BS of
        bsPicture: FEngine.Background.Picture.Assign(BackPicture);
        bsColor: begin
          C:= BackColor;
          FEngine.Background.Color:= C;
        end;
      end;
      FEngine.Background.Setting:= BS;
    end;
  end;
end;

//------------------------------------------------------------------------------

function TMainFrm.AddBackFile(const FileName, Alias: String): Integer;
begin with FBackFiles do begin
  Add(FileName + '=' + Alias);
  CustomSort(CompareBackFiles);
  Result:= IndexOfName(FileName);
end; end;

//------------------------------------------------------------------------------

procedure TMainFrm.SortBackFilesByDisplayName;
begin
  FBackFiles.CustomSort(CompareBackFiles);
end;

//------------------------------------------------------------------------------

procedure TMainFrm.pmiAboutBlankClick(Sender: TObject);
begin
  LoadRes('DefaultImage'); // picture with all special colors, useful for testing
  FFileName:= EmptyStr;
  DoCaption('About:Blank');
end;

//------------------------------------------------------------------------------

procedure TMainFrm.pmiAboutTestClick(Sender: TObject);
begin
  LoadRes('TestImage'); // picture with all special colors, useful for testing
  FFileName:= EmptyStr;
  DoCaption('About:Test');
end;

//------------------------------------------------------------------------------

procedure TMainFrm.pmiHelpClick(Sender: TObject);
begin
  //ConditionalBeep(QuestionBeep);
  if Application.MessageBox(PChar(_('Open web page with help?')), //t+
      PChar(Translate(sscMsgCapProceed)),
      MB_YESNO + MB_ICONQUESTION) = IDYES
  then
    WinOpen(Handle, _('http://vs.simutrans.com/shades_help/')); //t+
    // translatable so that it can be updated even if the engine is abandoned :(
end;

//------------------------------------------------------------------------------

procedure TMainFrm.FormResize(Sender: TObject);
begin
  if FTrackSize then with FOptions do begin
    Window.Top:= Self.Top;
    Window.Left:= Self.Left;
    Window.Height:= Self.Height;
    Window.Width:= Self.Width;
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.ConditionalBeep(const BeepType: Cardinal);
begin
   if FOptions.InterfaceOptions.Sounds then MessageBeep(BeepType);
end;

//------------------------------------------------------------------------------

function TMainFrm.GetViewportMidpoint: TPoint;
begin
  with Result, gOut do begin
    X:= Width div 2;
    Y:= Height div 2;
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.gOutMouseMove(Sender: TObject; Shift: TShiftState; X,
  Y: Integer; Layer: TCustomLayer);
var P: TPoint;
begin
  P:= Point(X, Y);
  P:= gOut.ControlToBitmap(P); // now translate to picture coords
  SetIndicator(P); // update info
  if Assigned(FTool) and (FTool is TMouseToolBase) then
    TMouseToolBase(FTool).MouseMove(P.X, P.Y, Shift);
  // propagate
end;

//------------------------------------------------------------------------------

procedure TMainFrm.tbtnFixBackgroundClick(Sender: TObject);
// fix backgroud color
var Tool: TFixBkTool;
begin
  Tool:= CreateTool(TFixBkTool) as TFixBkTool;
  Tool.Act;
  FreeAndNil(Tool);
end;

//------------------------------------------------------------------------------

procedure TMainFrm.gOutMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer; Layer: TCustomLayer);
var P: TPoint;
begin
  if Assigned(FTool) and (FTool is TClickToolBase) then begin
    P:= gOut.ControlToBitmap(Point(X, Y));
    TClickToolBase(FTool).MouseDown(Button, P.X, P.Y, Shift);
  end;
  // first propagate to tool, then handle zooming and panning viewport
  if Button = mbMiddle then begin
    if Shift = [ssMiddle] then begin
      Zoom(2); // with nothing it's IN
    end else if ssShift in Shift then begin
      Zoom(1/2); // with shift OUT
      // watch out for eventual type conversions - here 1/2=0.5 so OK
    end else if ssCtrl in Shift then begin
      gOut.ScrollToCenter(P.X, P.Y); // with ctrl PAN aka CENTER
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.gOutMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer; Layer: TCustomLayer);
var P: TPoint;
begin
  if Assigned(FTool) and (FTool is TMouseToolBase) then begin
    P:= gOut.ControlToBitmap(Point(X, Y));
    TMouseToolBase(FTool).MouseUp(Button, P.X, P.Y, Shift);
    // hard typecast might be more efficient and the check in IF took care of errors
  end;
end;

//------------------------------------------------------------------------------

function TMainFrm.CreateTool(const AToolType: TToolClass): TToolBase;
begin
  Result:= AToolType.Create(FEngine);
end;

procedure TMainFrm.UseTool(const AToolType: TToolClass);
begin
  if not (FTool is AToolType) then begin
    if Assigned(FTool) then FTool.Cancel;
    FreeAndNil(FTool);
    FTool:= CreateTool(AToolType);
  end;
end;

//------------------------------------------------------------------------------
// buttons switching mode

procedure TMainFrm.tbtnPenClick(Sender: TObject);
begin
  UseTool(TPaintPencilTool);
end;

procedure TMainFrm.tbtnFloodFillClick(Sender: TObject);
begin
  UseTool(TFillTool);
end;

procedure TMainFrm.tbtnPxEraserClick(Sender: TObject);
begin
  UseTool(TMagicPencilTool);
end;

procedure TMainFrm.tbtnFillEraserClick(Sender: TObject);
begin
  UseTool(TMagicFillTool);
end;

procedure TMainFrm.tbtnMoveClick(Sender: TObject);
begin
  UseTool(TMoverTool);
end;

procedure TMainFrm.tbtnBrushEraserClick(Sender: TObject);
begin
  UseTool(TMagicBrushTool);
end;

procedure TMainFrm.tbtnColorizeClick(Sender: TObject);
begin
  UseTool(TColorizePencilTool);
end;

//------------------------------------------------------------------------------
// propagate tool options from gui to tool instance

procedure TMainFrm.CbFramecbNightClick(Sender: TObject);
begin
  FEngine.ToolOptions.RemoveGlowing:= CbFrame.cbNight.Checked;
end;

procedure TMainFrm.CbFramecbPlayerClick(Sender: TObject);
begin
  FEngine.ToolOptions.RemovePlayer:= CbFrame.cbPlayer.Checked;
end;

procedure TMainFrm.cbFillDiagClick(Sender: TObject);
begin
  FEngine.ToolOptions.DiagonalFill:= CbFrame.cbFillDiag.Checked;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.tbtnCleanAllClick(Sender: TObject);
// replace all special color pixels
var Tool: TToolBase;
begin
  Tool:= CreateTool(TCleanAllTool);
  (Tool as TStaticToolBase).Act;
  FreeAndNil(Tool);
end;

//------------------------------------------------------------------------------

procedure TMainFrm.PictureChanged(Sender: TObject);
begin
  FPictureModified:= True;
  tbtnSave.Enabled:= True;
  with FEngine.UndoEngine do begin
    tbtnUndo.Enabled:= not AtStart;
    tbtnRedo.Enabled:= not AtEnd;
  end;
end;

//------------------------------------------------------------------------------

procedure TMainFrm.tbtnUndoClick(Sender: TObject);
begin
  FEngine.UndoEngine.Undo;
  PictureChanged(Self);
end;

procedure TMainFrm.tbtnRedoClick(Sender: TObject);
begin
  FEngine.UndoEngine.Redo;
  PictureChanged(Self);
end;

//------------------------------------------------------------------------------

procedure TMainFrm.BrushChanged(Sender: TObject);
begin
  FEngine.ToolOptions.Brush:= bsbtnBrushSel.SelectedBrush;
  if FTool is TMagicBrushTool then (FTool as TMagicBrushTool).BrushUpdated;
end;

//==============================================================================
// Now more class-model level stuff

procedure TMainFrm.WMEraseBkgnd(var Msg: TWMEraseBkgnd);
begin
  Msg.Result:= 1;
  // leave out inherited - do not redraw
end;

//------------------------------------------------------------------------------

procedure TMainFrm.WMSysCommand(var Message: TWMSysCommand);
begin
  inherited;
  with FOptions do case (Message.CmdType and $FFF0) of
    SC_MAXIMIZE: begin
      FTrackSize:= False;
      Window.State:= wsMaximized;
    end;
    SC_RESTORE: begin;
      FTrackSize:= True;
      FOptions.Window.State:= wsNormal;
      Window.Top:= Self.Top;
      Window.Left:= Self.Left;
    end;
  end;
end;

//------------------------------------------------------------------------------

constructor TMainFrm.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FEngine:= TPictureEngine.Create;

  FBackFiles:= TStringList.Create;
  FChunkList:= TList.Create;
  // OOP stuff

  with FBackFiles do begin
    Duplicates:= dupIgnore;
    CaseSensitive:= False;
    NameValueSeparator:= '=';
  end;

  FclBtnFace32:= Color32(clBtnFace);
  FclBtnText32:= Color32(clBtnText);
  FclHighlight32:= Color32(clHighlight);
  FclHighlightText32:= Color32(clHighlightText);
  // initialize colors

  FArrowNormal:= TBitmap32.Create;
  with FArrowNormal do begin
    SetSize(5, 5);
    DrawMode:= dmBlend;
    Clear($00FFFFFF); // rest is fully transparent
    LineT(2, 0, 2, 5, FclBtnText32); // arrow parts:  |
    LineT(0, 2, 2, 4, FclBtnText32);  //              \
    LineT(4, 2, 2, 4, FclBtnText32); //               /
  end;
  FArrowHighlight:= TBitmap32.Create;
  with FArrowHighlight do begin
    SetSize(5, 5);
    DrawMode:= dmBlend;
    Clear($00FFFFFF);
    LineT(2, 0, 2, 5, FclHighlightText32);
    LineT(0, 2, 2, 4, FclHighlightText32);
    LineT(4, 2, 2, 4, FclHighlightText32);
  end;

  FSelectedColor:= SpecColorArray[ScaDarkStart].Normal;
  FTrackSize:= True;
  FOldClrSelOverIndex:= -1;
end;

//------------------------------------------------------------------------------

destructor TMainFrm.Destroy;
begin
  FArrowNormal.Free;
  FArrowHighlight.Free;
  FBackFiles.Free;
  ClearChunkList(FChunkList);
  FChunkList.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

initialization
  AppPath:= ExtractFilePath(Application.ExeName);
  // where is program
end.
