(*************************************************************************
 *  ToolMagicBrushU.pas                                                  *
 *  Vladimr Slvik 2009-10                                              *
 *  Delphi 7 Personal                                                    *
 *  cp1250                                                               *
 *                                                                       *
 *  Magic pencil tool - removes special colours by painting over them.   *
 *                                                                       *
 *  -additional libraries: Graphics32                                    *
 *************************************************************************)

unit ToolMagicBrushU;

{$INCLUDE ..\Switches.inc}
{t default -}

//------------------------------------------------------------------------------

interface

uses Classes, ClassBaseU, GR32, Controls;

//------------------------------------------------------------------------------

type TMagicBrushTool = class(TMouseToolBase)
     private
       FPreviousPos: TRect;
       FBrushMask: TBitmap32;
       FBrushOverlay: TBitmap32;
       FWorkPlace: TBitmap32;
       FWorkMask: TBitmap32;
       function MouseToBrush(const X, Y: Integer): TRect;
       procedure SetBrush(const NewBrush: TBitmap32);
     public
       constructor Create(AEngine: TPictureEngineBase); override;
       destructor Destroy; override;
       procedure MouseMove(const X, Y: Integer; const Shift: TShiftState);
         override;
       procedure MouseDown(const Button: TMouseButton; const X, Y: Integer;
         const Shift: TShiftState); override;
       procedure MouseUp(const Button: TMouseButton; const X, Y: Integer;
         const Shift: TShiftState); override;
       procedure MouseLeave; override;
       procedure Cancel; override;
       procedure BrushUpdated;
       property Brush: TBitmap32 write SetBrush;
     end;

//==============================================================================
implementation

uses CoreLowU, SysUtils, CoreTypeU, CoreEngineU, CalcUtilU, GR32_Blend;
//==============================================================================

constructor TMagicBrushTool.Create(AEngine: TPictureEngineBase);
begin
  inherited Create(AEngine);
  FBrushMask:= TBitmap32.Create;
  FBrushOverlay:= TBitmap32.Create;
  FWorkPlace:= TBitmap32.Create;
  FWorkmask:= TBitmap32.Create;
  FPreviousPos:= ZeroRect;
  SetBrush(FEngine.ToolOptions.Brush);
end;

//------------------------------------------------------------------------------

destructor TMagicBrushTool.Destroy;
begin
  if FActive then Cancel else begin
    FEngine.RedrawRect(FPreviousPos);
    FEngine.UpdateRect(FPreviousPos);
  end;;
  FBrushMask.Free;
  FBrushOverlay.Free;
  FWorkPlace.Free;
  FWorkMask.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

function TMagicBrushTool.MouseToBrush(const X, Y: Integer): TRect;
begin with Result, FBrushMask do begin
  Left:= X - Width div 2;
  Top:= Y - Height div 2;
  Right:= Left + Width;
  Bottom:= Top + Height;
end; end;

//------------------------------------------------------------------------------

procedure TMagicBrushTool.SetBrush(const NewBrush: TBitmap32);
begin
  if FActive then raise Exception.Create(
    'Error: TMagicBrushTool.SetBrush: Can''t change brush when active!');
  (*
      Table of bitmaps, colours and usages:

   Image name           | Brush cl  | Around cl | Notes
  ======================+===========+===========+===============================
   NewBrush             | FF 000000 | FF FFFFFF | Input - read from resources
  ----------------------+-----------+-----------+-------------------------------
   FBrushMask           | FF FFFFFF | 00 ****** | Mask for CleanByMask - white
                        |           | 00 000000 | colour chosen there "just because"
  ----------------------+-----------+-----------+-------------------------------
   FBrushOverlay        | ** ****** | 00 ****** | Drawn over picture on viewport,
                        | user set  | 00 000000 | must be fully transparent
                        |           |           | outside brush shape
  ----------------------+-----------+-----------+-------------------------------
   FWorkPlace           |                       | Processing pieces of picture
                        |      not present      | under brush takes place there,
                        |                       | brush shape is not present.
  ----------------------+-----------+-----------+-------------------------------
   FWorkMask            | 00 ****** | FF E7FFFF | This mask is drawn over FWorkPlace
                        | 00 000000 |           | to hide the outside-brush area.
                        |           | indirect  | With FWorkPlace, last step is
                        |           | result as | making its new background 
                        |           | 00 E7FFFF | transparent.
  ----------------------+-----------+-----------+-------------------------------
                        |           |           |
  
   Asterisks * indicate that any value is usable; row below lists used value.
  *)
  FBrushMask.Assign(NewBrush);
  ReplaceColor(FBrushMask, $FFFFFFFF, $00000000, FBrushMask.BoundsRect); // white to transparency
  ReplaceColor(FBrushMask, $FF000000, $FFFFFFFF, FBrushMask.BoundsRect); // black to white
  FBrushOverlay.Assign(FBrushMask);
  // transparent can stay as is
  ReplaceColor(FBrushOverlay, $FFFFFFFF, $8000FF00, FBrushOverlay.BoundsRect); // white to shape
  {$MESSAGE WARN 'TODO: add user choice of brush colour'}
  FWorkMask.Assign(NewBrush);
  ReplaceColor(FWorkMask, $FFFFFFFF, BackClrOpaque, FWorkMask.BoundsRect); // white to background
  ReplaceColor(FWorkMask, $FF000000, $00000000, FWorkMask.BoundsRect); // black to transparency
  FWorkPlace.SetSizeFrom(FBrushMask);
end;

//------------------------------------------------------------------------------

procedure TMagicBrushTool.MouseMove(const X, Y: Integer; const Shift: TShiftState);
var R: TRect;
begin
  // Apart from the colour insanity dcescribed above, do watch out for DrawMode,
  // or it will bite back somewhere and make everything stop working.
  R:= MouseToBrush(X, Y);
  FEngine.RedrawRect(FPreviousPos); // 
  if FActive then try
    // active - remove under brush
    Assert(Assigned(FEngine));

    FWorkPlace.DrawMode:= dmOpaque;
    FEngine.Picture.DrawMode:= dmOpaque;
    FWorkPlace.Draw(0, 0, R, FEngine.Picture); // copy true data

    with FEngine.ToolOptions do CleanByMask(FBrushMask, FWorkPlace, RemoveGlowing,
      RemovePlayer); // clean as desired (tool action seen by user)

    FWorkMask.DrawMode:= dmBlend;
    FWorkPlace.DrawMode:= dmBlend;
    FWorkPlace.Draw(0, 0, FWorkMask); // mask out region outside brush with opaque colour

    ReplaceColor(FWorkPlace, BackClrOpaque, BackClrTransparent, FWorkPlace.BoundsRect); // and then mask it from drawing

    FEngine.Overlay.DrawMode:= dmBlend;
    FWorkPlace.DrawTo(FEngine.Overlay, R); // commit this step to overlay

    FEngine.UpdateRect(R);
    FEngine.RedrawRect(R);
    FChangeArea:= AccumulateRect(FChangeArea, R);
  except
    // eat aborts
  end;
  FBrushOverlay.DrawMode:= dmBlend;
  FBrushOverlay.DrawTo(FEngine.Viewport.Bitmap, R);
  FPreviousPos:= R;
end;

//------------------------------------------------------------------------------

procedure TMagicBrushTool.MouseDown(const Button: TMouseButton;
    const X, Y: Integer; const Shift: TShiftState);
begin
  if (not FActive) and (Button = mbLeft) then begin 
    Assert(Assigned(FEngine));
    FActive:= True;
    Self.MouseMove(X, Y, Shift); // paint the first pixel
  end else if FActive and (Button = mbRight) then begin
    // stop activity
    Cancel;
  end;
end;

//------------------------------------------------------------------------------

procedure TMagicBrushTool.MouseUp(const Button: TMouseButton;
    const X, Y: Integer; const Shift: TShiftState);
begin
  // note: final redraw also takes care of brush still visible in viewport,
  // no special handling needed
  if FActive and (Button = mbLeft) then begin
    FActive:= False;
    FEngine.CommitOverlay(FChangeArea);
    FEngine.RedrawRect(FChangeArea);
    FChangeArea:= ZeroRect;
  end;
end;

//------------------------------------------------------------------------------

procedure TMagicBrushTool.MouseLeave; {$I EMPTY.inc}
// do not react, keep working

//------------------------------------------------------------------------------

procedure TMagicBrushTool.Cancel;
begin
  // final redraw again handles last viewport brush draw
  if FActive then begin
    FActive:= False;
    FEngine.DiscardOverlay;
    FEngine.UpdateRect(FChangeArea);
    FEngine.RedrawRect(FChangeArea);
    FChangeArea:= ZeroRect;
  end;
end;

//------------------------------------------------------------------------------

procedure TMagicBrushTool.BrushUpdated;
begin
  SetBrush(FEngine.ToolOptions.Brush);
end;

//------------------------------------------------------------------------------

end.
