(*************************************************************************
 *  ToolMoverU.pas                                                       *
 *  Vladimr Slvik 2009-10                                              *
 *  Delphi 7 Personal                                                    *
 *  cp1250                                                               *
 *                                                                       *
 *  Moving tool - moves separate parts of picture to new position.       *
 *                                                                       *
 *  -additional libraries: Graphics32                                    *
 *************************************************************************)

unit ToolMoverU;

{$INCLUDE ..\Switches.inc}
{t default -}

//------------------------------------------------------------------------------

interface

uses Classes, ClassBaseU, GR32, Controls;

//------------------------------------------------------------------------------

type TMoverTool = class(TMouseToolBase)
     private
       FSprite: TBitmap32;
       FSpriteClickOffset: TPoint;
       FSpriteOriginalPos: TPoint;
       FPreviousPos: TRect;
       procedure Pick(const X, Y: Integer);
       procedure Drop(const X, Y: Integer);
     public
       constructor Create(AEngine: TPictureEngineBase); override;
       destructor Destroy; override;
       procedure MouseMove(const X, Y: Integer; const Shift: TShiftState); override;
       procedure MouseDown(const Button: TMouseButton; const X, Y: Integer;
         const Shift: TShiftState); override;
       procedure MouseUp(const Button: TMouseButton; const X, Y: Integer;
         const Shift: TShiftState); override;
       procedure MouseLeave; override;
       procedure Cancel; override;
     end;

//==============================================================================
implementation

uses CoreLowU, SysUtils, CoreTypeU, CoreEngineU, CalcUtilU, GR32_Blend;
//==============================================================================

constructor TMoverTool.Create(AEngine: TPictureEngineBase);
begin
  inherited Create(AEngine);
  FSprite:= TBitmap32.Create;
  FPreviousPos:= ZeroRect;
end;

//------------------------------------------------------------------------------

destructor TMoverTool.Destroy;
begin
  inherited Destroy;
  FSprite.Free;
end;

//------------------------------------------------------------------------------

procedure TMoverTool.MouseDown(const Button: TMouseButton; const X, Y: Integer;
  const Shift: TShiftState);
begin
  case Button of
    mbLeft: if FActive then Drop(X, Y) else Pick(X, Y);
    mbRight: Cancel;
  end;
end;

//------------------------------------------------------------------------------

procedure TMoverTool.Pick(const X, Y: Integer);
var ClickPos: TPoint;
    Mask: TBitmap32;
    SpriteRect: TRect;
begin
  (* This routine might need some more optimization since working with
     picture-sized mask is quite clumsy and eats resources; in fact, the mask
     might come out of GetImagePart already cropped to the interesting rectangle
     and speed up the whole process twice as much. But this is a rare action and
     hopefully won't cause problems. *)
  // PS: reading this after a few months, I have no idea why it should be so :-(   
  FSprite.DrawMode:= dmBlend;
  ClickPos:= Point(X, Y);
  Mask:= TBitmap32.Create;
  if GetImagePart(FEngine.Picture, ClickPos, Mask, SpriteRect) then begin
    // PieceRect is now a rectangle describing the position of picked item. Mask
    // has size of original picture, is filled with opaque background and has
    // transparent hole where the sprite is.
    FSpriteOriginalPos:= SpriteRect.TopLeft;
    // save offset of sprite from picture
    FSprite.SetSize(Abs(RectWidth(SpriteRect)), Abs(RectHeight(SpriteRect)));
    FSpriteClickOffset:= SubtractPoints(ClickPos, SpriteRect.TopLeft);
    // save offset of cursor within sprite (need to know or it would jump when picked)
    FSprite.Draw(0, 0, SpriteRect, FEngine.Picture);
    // first copy part with sprite from big picture
    FSprite.Draw(0, 0, SpriteRect, Mask);
    // now cover non-sprite with background colour from mask
    MakeTransparent(FSprite, BackClrOpaque, FSprite.BoundsRect);
    // finally turn sprite's background to transparent again -> is directly drawable
    InvertMask(Mask);
    // flip mask so that now it covers the one part and rest can be seen through
    FEngine.Overlay.Draw(SpriteRect, SpriteRect, Mask);
    // and copy that to overlay to make the sprite disappear for user
    FEngine.UpdateRect(SpriteRect);
    // This call can not be omitted since overlay itself MUST be applied NOW,
    // even if this place is redrawn immediately!
    MouseMove(X, Y, []);
    (* Now the part/sprite/item completely disappeared from viewport, hidden by
       opaque overlay and redraw, but the next moment mouse moves and it will
       come back from oblivion. That is bad because it makes flicker when picking
       things. So, call immediately false move on the same place, which will draw
       it again - in the same place. it's too fast to notice so from user's
       perspective nothing happens at all. *)
    // Shift state does not matter for this tool so we can get away with such
    // sloppy code inventing fake parameter values ;-)
    FActive:= True;
  end;
  Mask.Free;
end;

//------------------------------------------------------------------------------

procedure TMoverTool.Drop(const X, Y: Integer);
var ClickPos: TPoint;
    OriginalArea: TRect;
    TargetArea: TRect;
    AffectedArea: TRect;
begin
  FSprite.DrawMode:= dmBlend;
  ClickPos:= Point(X, Y);
  OriginalArea:= FSprite.BoundsRect;
  with FSpriteOriginalPos do OffsetRect(OriginalArea, X, Y);
  // original area = offset + sprite 
  TargetArea:= FSprite.BoundsRect;
  with ClickPos do OffsetRect(TargetArea, X, Y);
  with FSpriteClickOffset do OffsetRect(TargetArea, -X, -Y);
  // target area = sprite + clicked - click offset
  UnionRect(AffectedArea, OriginalArea, TargetArea);
  // affected area = source + target
  FEngine.Overlay.DrawMode:= dmBlend;
  FSprite.DrawMode:= dmBlend;
  with TargetArea do FEngine.Overlay.Draw(Left, Top, FSprite);
  // drop the sprite back on picture; should work seamlessly since sprite has
  // transparent background; must be blended draw since sprite must not replace
  // previously covered area of "pick" - prime use of this tool is realigning,
  // where old & new positions overlap!
  if RectArea(AffectedArea) < 2 * RectArea(OriginalArea) then begin
    // decide - is it less work to use union or two separate parts?
    FEngine.CommitOverlay(AffectedArea);
    FEngine.RedrawRect(AffectedArea);
  end else with FEngine do begin
    FEngine.CommitOverlay([OriginalArea, TargetArea]);
    RedrawRect(OriginalArea);
    RedrawRect(TargetArea);
  end;
  FActive:= False;
end;

//------------------------------------------------------------------------------

procedure TMoverTool.MouseMove(const X, Y: Integer; const Shift: TShiftState);
var ActualPos: TRect;
begin
  if FActive then begin
    ActualPos:= FSprite.BoundsRect;
    OffsetRect(ActualPos, X, Y);
    with FSpriteClickOffset do OffsetRect(ActualPos, -X, -Y);
    // place = sprite + mouse - start_mouse_offset
    if not RIs0(FPreviousPos) then FEngine.RedrawRect(FPreviousPos);
    // erase from previous position
    with ActualPos do FEngine.Viewport.Bitmap.Draw(Left, Top, FSprite);
    // draw in new position
    FPreviousPos:= ActualPos;
    // keep track of position
  end;
end;

//------------------------------------------------------------------------------

procedure TMoverTool.Cancel;
var OriginalArea: TRect;
    AffectedArea: TRect;
begin
  if FActive then begin
    FSprite.DrawMode:= dmBlend;
    OriginalArea:= FSprite.BoundsRect;
    with FSpriteOriginalPos do OffsetRect(OriginalArea, X, Y);
    // original area = offset + sprite
    UnionRect(AffectedArea, OriginalArea, FPreviousPos);
    // affected area = source + target
    FEngine.DiscardOverlay;
    // throw all away
    if RectArea(AffectedArea) < 2 * RectArea(OriginalArea) then begin
      // decide - is it less work to use union or two separate parts?
      FEngine.UpdateRect(AffectedArea);
      FEngine.RedrawRect(AffectedArea);
    end else with FEngine do begin
      UpdateRect(OriginalArea);
      UpdateRect(FPreviousPos);
      RedrawRect(OriginalArea);
      RedrawRect(FPreviousPos);
    end;
    FActive:= False;
    FPreviousPos:= ZeroRect;
  end;
end;

//------------------------------------------------------------------------------

procedure TMoverTool.MouseUp(const Button: TMouseButton; const X, Y: Integer;
  const Shift: TShiftState); {$I ..\EMPTY.inc}

//------------------------------------------------------------------------------

procedure TMoverTool.MouseLeave; {$I ..\EMPTY.inc}

//------------------------------------------------------------------------------

end.
